<?php
/**
 * Object Cache Support
 *
 * Detects and manages persistent object cache (Redis, Memcached)
 * and registers global cache groups for multisite compatibility.
 *
 * @package Mamba\Support
 * @since   1.0.0
 */

namespace Mamba\Support;

/**
 * Class ObjectCache
 *
 * Provides object cache detection, type identification, and global group
 * registration for multisite environments with Redis or Memcached.
 *
 * @since 1.0.0
 */
final class ObjectCache {
    
    /**
     * Check if persistent object cache is available
     */
    public static function hasPersistentCache(): bool {
        // Safety check for WordPress functions
        if (!function_exists('wp_using_ext_object_cache')) {
            return false;
        }
        
        $result = wp_using_ext_object_cache();
        
        // Ensure we always return a boolean
        return $result === true;
    }
    
    /**
     * Get cache type for display
     */
    public static function getCacheType(): string {
        if (!self::hasPersistentCache()) {
            return 'Not detected (using file fallback)';
        }
        
        // Try to detect specific cache type
        if (class_exists('Redis') && function_exists('wp_cache_get_redis_client')) {
            return 'Redis';
        }
        
        if (class_exists('Memcached') || class_exists('Memcache')) {
            return 'Memcached';
        }
        
        return 'Object Cache';
    }
    
    /**
     * Get detailed cache backend information
     */
    public static function getBackendInfo(): array {
        $info = [
            'persistent' => self::hasPersistentCache(),
            'type' => self::getCacheType(),
            'backends' => []
        ];
        
        // Check what backends are actually being used
        if (self::hasPersistentCache()) {
            // Persistent object cache is available
            if (class_exists('Redis') && function_exists('wp_cache_get_redis_client')) {
                $info['backends']['store_api'] = 'redis';
                $info['backends']['tags'] = 'redis';
                $info['backends']['stats'] = 'redis';
                $info['backends']['cart_fragments'] = 'redis';
            } elseif (class_exists('Memcached') || class_exists('Memcache')) {
                $info['backends']['store_api'] = 'memcached';
                $info['backends']['tags'] = 'memcached';
                $info['backends']['stats'] = 'memcached';
                $info['backends']['cart_fragments'] = 'memcached';
            } else {
                $info['backends']['store_api'] = 'object_cache';
                $info['backends']['tags'] = 'object_cache';
                $info['backends']['stats'] = 'object_cache';
                $info['backends']['cart_fragments'] = 'object_cache';
            }
        } else {
            // Using file fallback for all cache groups
            $info['backends']['store_api'] = 'file';
            $info['backends']['tags'] = 'file';
            $info['backends']['stats'] = 'file';
            $info['backends']['cart_fragments'] = 'file';
        }
        
        return $info;
    }
    
    /**
     * Register global groups for multisite compatibility
     */
    public static function registerGlobalGroups(): void {
        // Safety check for WordPress functions
        if (!function_exists('wp_cache_add_global_groups')) {
            return;
        }
        
        wp_cache_add_global_groups([
            'mamba_store_api',
            'mamba_tags', 
            'mamba_stats',
            'mamba_cart_fragments'
        ]);
    }
}
