<?php if (!defined('ABSPATH')) exit; ?>

<?php
// Get current settings and stats
$optimizer = new \Mamba\Modules\DB\Services\DatabaseOptimizer();
$tasks = $optimizer->getTasks();
$settings = get_option('mamba_db_optimizer_settings', []);
$counts = $optimizer->calculateCounts();

// Calculate overall stats
$totalTasks = count($tasks);
$enabledTasks = 0;
$totalAffected = 0;
$lastRunTime = 0;
$successfulRuns = 0;

foreach ($tasks as $taskId => $task) {
    $taskSettings = $settings[$taskId] ?? [];
    if (!empty($taskSettings['enabled'])) {
        $enabledTasks++;
    }
    if (!empty($taskSettings['last'])) {
        $lastRunTime = max($lastRunTime, $taskSettings['last']);
        if (!empty($taskSettings['last_rows'])) {
            $successfulRuns++;
            $totalAffected += $taskSettings['last_rows'];
        }
    }
}

$interval_labels = [
    'every_6_hours' => __('Every 6 hours', 'mamba-cache-for-woocommerce'),
    'every_12_hours' => __('Every 12 hours', 'mamba-cache-for-woocommerce'),
    'daily' => __('Daily', 'mamba-cache-for-woocommerce'),
    'every_2_days' => __('Every 2 days', 'mamba-cache-for-woocommerce'),
    'weekly' => __('Weekly', 'mamba-cache-for-woocommerce')
];
?>

<div class="mamba-admin-wrap">
    <?php
    // Notices
    if (!empty($_GET['saved'])) {
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( __('Settings saved successfully. Schedules updated.', 'mamba-cache-for-woocommerce') ) . '</p></div>';
    }
    if (!class_exists('WooCommerce')) {
        echo '<div class="notice notice-warning"><p><strong>' . esc_html( __('WooCommerce is not active.', 'mamba-cache-for-woocommerce') ) . '</strong> ' . esc_html( __('You can configure here; tasks run only when Woo is active.', 'mamba-cache-for-woocommerce') ) . '</p></div>';
    }
    if (defined('DISABLE_WP_CRON') && DISABLE_WP_CRON) {
        echo '<div class="notice notice-warning"><p><strong>' . esc_html( __('WP-Cron is disabled.', 'mamba-cache-for-woocommerce') ) . '</strong> ' . esc_html( __('Automation won\'t run until it is enabled or a real cron triggers wp-cron.php.', 'mamba-cache-for-woocommerce') ) . '</p></div>';
    }
    ?>

    <!-- Database Optimization Overview -->
    <div class="mamba-form-section">
        <div class="mamba-section-header">
            <h4 class="mamba-section-title">
                <span class="dashicons dashicons-database"></span>
                <?php echo esc_html( __('Database Optimization Overview', 'mamba-cache-for-woocommerce') ); ?>
            </h4>
        </div>
        
        <div class="mamba-db-overview">
            <div class="mamba-db-card">
                <div class="mamba-db-icon">
                    <span class="dashicons dashicons-performance"></span>
                </div>
                <div class="mamba-db-content">
                    <h3 class="mamba-db-title"><?php echo esc_html( __('Database Maintenance', 'mamba-cache-for-woocommerce') ); ?></h3>
                    <p class="mamba-db-description">
                        <?php echo esc_html( __('Automatically clean and optimize your WooCommerce database to improve performance, reduce storage usage, and maintain data integrity through scheduled maintenance tasks.', 'mamba-cache-for-woocommerce') ); ?>
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Execution Overview Section -->
    <div class="mamba-form-section">
        <div class="mamba-section-header">
            <h4 class="mamba-section-title">
                <span class="dashicons dashicons-clock"></span>
                <?php echo esc_html( __('Scheduled Execution Status', 'mamba-cache-for-woocommerce') ); ?>
            </h4>
        </div>
        
        <div class="mamba-optimization-intro">
                <p class="mamba-optimization-description">
                <?php echo esc_html( __('Monitor your database optimization tasks and their execution history.', 'mamba-cache-for-woocommerce') ); ?>
            </p>
        </div>
        
        <div class="mamba-stats-grid">
            <div class="mamba-stat-card">
                <div class="mamba-stat-header">
                    <span class="dashicons dashicons-admin-tools"></span>
                    <h4><?php echo esc_html( __('Scheduled Tasks', 'mamba-cache-for-woocommerce') ); ?></h4>
                </div>
                <div class="mamba-stat-value"><?php echo esc_html($enabledTasks); ?>/<?php echo esc_html($totalTasks); ?></div>
                <div class="mamba-stat-description"><?php echo esc_html( __('Active optimization tasks', 'mamba-cache-for-woocommerce') ); ?></div>
            </div>
            
            <div class="mamba-stat-card">
                <div class="mamba-stat-header">
                    <span class="dashicons dashicons-calendar-alt"></span>
                    <h4><?php echo esc_html( __('Last Cleanup', 'mamba-cache-for-woocommerce') ); ?></h4>
                </div>
                <div class="mamba-stat-value">
                    <?php echo $lastRunTime ? esc_html(date_i18n('M j, g:i A', $lastRunTime)) : esc_html( __('Never', 'mamba-cache-for-woocommerce') ); ?>
                </div>
                <div class="mamba-stat-description"><?php echo esc_html( __('Most recent execution', 'mamba-cache-for-woocommerce') ); ?></div>
            </div>
            
            <div class="mamba-stat-card">
                <div class="mamba-stat-header">
                    <span class="dashicons dashicons-chart-bar"></span>
                    <h4><?php echo esc_html( __('Total Cleaned', 'mamba-cache-for-woocommerce') ); ?></h4>
                </div>
                <div class="mamba-stat-value"><?php echo esc_html(number_format_i18n($totalAffected)); ?> rows</div>
                <div class="mamba-stat-description"><?php echo esc_html( __('Database rows processed', 'mamba-cache-for-woocommerce') ); ?></div>
            </div>
            
            <div class="mamba-stat-card">
                <div class="mamba-stat-header">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <h4><?php echo esc_html( __('Successful Runs', 'mamba-cache-for-woocommerce') ); ?></h4>
                </div>
                <div class="mamba-stat-value"><?php echo esc_html($successfulRuns); ?></div>
                <div class="mamba-stat-description"><?php echo esc_html( __('Completed optimizations', 'mamba-cache-for-woocommerce') ); ?></div>
            </div>
        </div>
    </div>

    <!-- Task Management Section -->
    <div class="mamba-form-section">
        <div class="mamba-section-header">
                <h4 class="mamba-section-title">
                <span class="dashicons dashicons-admin-tools"></span>
                <?php echo esc_html( __('Task Management', 'mamba-cache-for-woocommerce') ); ?>
            </h4>
        </div>
        
        <div class="mamba-optimization-intro">
                <p class="mamba-optimization-description">
                <?php echo esc_html( __('Configure and monitor your database optimization tasks.', 'mamba-cache-for-woocommerce') ); ?>
            </p>
        </div>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('mamba_db_save'); ?>
            <input type="hidden" name="action" value="mamba_db_save" />

            <div class="mamba-tasks-grid">
                <?php foreach ($tasks as $taskId => $task): 
                    $taskSettings = $settings[$taskId] ?? [];
                    $args = $taskSettings['args'] ?? [];
                    $enabled = !empty($taskSettings['enabled']);
                    $interval = $taskSettings['interval'] ?? ($task['default_interval'] ?? 'weekly');
                    $lastTs = !empty($taskSettings['last']) ? intval($taskSettings['last']) : 0;
                    $lastRows = isset($taskSettings['last_rows']) ? intval($taskSettings['last_rows']) : null;
                    $current = isset($counts[$taskId]) ? number_format_i18n($counts[$taskId]) : '0';
                    
                    // Get next scheduled run
                    $nextRun = '—';
                    if ($enabled) {
                        $hook = 'mamba_db_task_' . $taskId;
                        $nextScheduled = wp_next_scheduled($hook, [$taskId]);
                        if ($nextScheduled) {
                            $nextRun = date_i18n('M j, g:i A', $nextScheduled);
                        }
                    }

                    // Set defaults for UI
                    if (isset($task['fields']['retention_days']) && empty($args['retention_days'])) {
                        $args['retention_days'] = $task['fields']['retention_days'][2] ?? 90;
                    }
                    if (isset($task['fields']['retention_hours']) && empty($args['retention_hours'])) {
                        $args['retention_hours'] = $task['fields']['retention_hours'][3] ?? 24;
                    }
                    if (isset($task['fields']['months']) && empty($args['months'])) {
                        $args['months'] = 6;
                    }
                    if (isset($task['fields']['statuses']) && empty($args['statuses'])) {
                        $args['statuses'] = $task['fields']['statuses'];
                    }
                ?>
                
                <div class="mamba-task-card">
                    <div class="mamba-task-header">
                        <div class="mamba-task-info">
                            <h5 class="mamba-task-name"><?php echo esc_html($task['label']); ?></h5>
                            <p class="mamba-task-description"><?php echo esc_html($task['desc']); ?></p>
                        </div>
                        <div class="mamba-task-status">
                            <?php 
                            // Premium DB tasks - must match backend list in DatabaseOptimizer.php
                            // FREE tasks (6): as_prune, post_revisions, expired_transients, spam_comments, trashed_comments, trashed_posts
                            // PREMIUM tasks (12): All WooCommerce-specific and advanced cleanup tasks
                            $premium_db_tasks = [
                                'sessions_expired',           // Clear expired sessions
                                'customer_sessions_orphans',  // Clean sessions for deleted users
                                'reserved_stock',             // Clear stale reserved stock
                                'order_lookup_orphans',       // Clean order lookup orphans
                                'order_meta_orphans',         // Clean orphaned order metadata
                                'download_log_rotate',        // Prune old download logs
                                'order_items_orphans',        // Clean orphaned order items
                                'system_notes_rotate',        // Remove old system order notes
                                'attr_terms_prune',           // Remove unused attribute terms
                                'product_meta_orphans',       // Clean orphaned product metadata
                                'draft_orders_cleanup',       // Delete draft checkout orders
                                'auto_drafts'                 // Delete auto-draft posts
                            ];
                            $is_premium_task = in_array($taskId, $premium_db_tasks, true) || !empty($task['premium']);
                            $is_locked = $is_premium_task && !mamba_fs()->can_use_premium_code__premium_only();
                            
                            if ($is_locked) {
                                $enabled = false; 
                            }
                            ?>
                            <div class="mamba-toggle-switch">
                                <input id="task_<?php echo esc_attr($taskId); ?>" 
                                       type="checkbox" 
                                       name="task[<?php echo esc_attr($taskId); ?>][enabled]" 
                                       value="1" 
                                       <?php checked($enabled); ?> 
                                       <?php disabled($is_locked); ?> />
                                <label for="task_<?php echo esc_attr($taskId); ?>"><?php echo esc_html( __('Enable', 'mamba-cache-for-woocommerce') ); ?></label>
                            </div>
                            <?php if ($is_locked): ?>
                                <span class="mamba-status-badge mamba-premium-badge-small" style="background: linear-gradient(135deg, #6366f1 0%, #4f46e5 100%); color: white;">
                                    <?php echo esc_html( __('PREMIUM', 'mamba-cache-for-woocommerce') ); ?>
                                </span>
                            <?php else: ?>
                                <span class="mamba-status-badge <?php echo esc_attr( $enabled ? 'mamba-status-success' : 'mamba-status-warning' ); ?>">
                                    <?php echo $enabled ? esc_html( __('Scheduled', 'mamba-cache-for-woocommerce') ) : esc_html( __('Disabled', 'mamba-cache-for-woocommerce') ); ?>
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="mamba-task-details">
                        <div class="mamba-db-detail-item">
                            <span class="mamba-detail-label"><?php echo esc_html( __('Schedule', 'mamba-cache-for-woocommerce') ); ?></span>
                            <select name="task[<?php echo esc_attr($taskId); ?>][interval]" class="mamba-select" <?php echo esc_attr( !$enabled ? 'disabled' : '' ); ?>>
                                <?php foreach ($interval_labels as $value => $label): ?>
                                    <option value="<?php echo esc_attr($value); ?>" <?php selected($interval, $value); ?>>
                                        <?php echo esc_html($label); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="mamba-db-detail-item">
                            <span class="mamba-detail-label"><?php echo esc_html( __('Next Run', 'mamba-cache-for-woocommerce') ); ?></span>
                            <span class="mamba-detail-value">
                                <?php echo $enabled ? esc_html($nextRun) : esc_html( __('Not scheduled', 'mamba-cache-for-woocommerce') ); ?>
                            </span>
                        </div>
                        
                        <div class="mamba-db-detail-item">
                            <span class="mamba-detail-label"><?php echo esc_html( __('Last Run', 'mamba-cache-for-woocommerce') ); ?></span>
                            <span class="mamba-detail-value">
                                <?php if ($lastTs): ?>
                                    <?php echo esc_html(date_i18n('M j, g:i A', $lastTs)); ?>
                                    <?php if ($lastRows !== null): ?>
                                        <?php /* translators: %s is the number of rows */ ?>
                                        <br><small><?php printf( esc_html( __('%s rows', 'mamba-cache-for-woocommerce') ), esc_html( number_format_i18n($lastRows) ) ); ?></small>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <?php echo esc_html( __('Never run', 'mamba-cache-for-woocommerce') ); ?>
                                <?php endif; ?>
                            </span>
                        </div>
                        
                        <div class="mamba-db-detail-item">
                            <span class="mamba-detail-label"><?php echo esc_html( __('Current', 'mamba-cache-for-woocommerce') ); ?></span>
                            <span class="mamba-detail-value mamba-current" data-task="<?php echo esc_attr($taskId); ?>">
                                <?php echo esc_html($current); ?> rows
                            </span>
                        </div>
                    </div>
                    
                    <div class="mamba-task-actions">
                        <button type="button" 
                                class="mamba-run-task" 
                                data-task="<?php echo esc_attr($taskId); ?>"
                                data-nonce="<?php echo esc_attr( wp_create_nonce('mamba_db_optimize') ); ?>">
                            <span class="dashicons dashicons-controls-play"></span>
                            <?php echo esc_html( __('Run Now', 'mamba-cache-for-woocommerce') ); ?>
                        </button>
                        
                        <?php if (!empty($task['fields'])): ?>
                            <button type="button" 
                                    class="mamba-settings-toggle" 
                                    data-task="<?php echo esc_attr($taskId); ?>">
                                <span class="dashicons dashicons-admin-generic"></span>
                                <?php echo esc_html( __('Settings', 'mamba-cache-for-woocommerce') ); ?>
                            </button>
                        <?php endif; ?>
                        
                        <span class="mamba-task-status" id="status-<?php echo esc_attr($taskId); ?>"></span>
                    </div>
                    
                    <!-- Settings Section (Collapsible) -->
                    <?php if (!empty($task['fields'])): ?>
                    <div class="mamba-task-settings" id="settings-<?php echo esc_attr($taskId); ?>" style="display: none;">
                        <div class="mamba-settings-content">
                            <h6><?php echo esc_html( __('Task Settings', 'mamba-cache-for-woocommerce') ); ?></h6>
                            <div class="mamba-settings-grid">
                                <?php if (!empty($task['fields']['retention_days'])): ?>
                                    <div class="mamba-setting-field">
                                        <label><?php echo esc_html( __('Delete entries older than', 'mamba-cache-for-woocommerce') ); ?></label>
                                        <select name="task[<?php echo esc_attr($taskId); ?>][args][retention_days]">
                                            <?php foreach ($task['fields']['retention_days'] as $days): ?>
                                                <option value="<?php echo esc_attr($days); ?>" <?php selected(intval($args['retention_days']), $days); ?>>
                                                    <?php printf( esc_html( __('%d days', 'mamba-cache-for-woocommerce') ), intval( $days ) ); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                <?php endif; ?>

                                <?php if (!empty($task['fields']['retention_hours'])): ?>
                                    <div class="mamba-setting-field">
                                        <label><?php echo esc_html( __('Delete entries older than', 'mamba-cache-for-woocommerce') ); ?></label>
                                        <select name="task[<?php echo esc_attr($taskId); ?>][args][retention_hours]">
                                            <?php foreach ($task['fields']['retention_hours'] as $hours): ?>
                                                <option value="<?php echo esc_attr($hours); ?>" <?php selected(intval($args['retention_hours'] ?? 24), $hours); ?>>
                                                    <?php printf( esc_html( _n('%d hour', '%d hours', $hours, 'mamba-cache-for-woocommerce') ), intval( $hours ) ); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                <?php endif; ?>

                                <?php if (!empty($task['fields']['months'])): ?>
                                    <div class="mamba-setting-field">
                                        <label><?php echo esc_html( __('Delete notes older than', 'mamba-cache-for-woocommerce') ); ?></label>
                                        <select name="task[<?php echo esc_attr($taskId); ?>][args][months]">
                                            <?php foreach ($task['fields']['months'] as $months): ?>
                                                <option value="<?php echo esc_attr($months); ?>" <?php selected(intval($args['months']), $months); ?>>
                                                <?php printf( esc_html( __('%d months', 'mamba-cache-for-woocommerce') ), intval( $months ) ); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                <?php endif; ?>

                                <?php if (!empty($task['fields']['statuses'])): ?>
                                    <div class="mamba-setting-field">
                                        <label><?php echo esc_html( __('Include statuses:', 'mamba-cache-for-woocommerce') ); ?></label>
                                        <div class="mamba-checkbox-group">
                                            <?php foreach ($task['fields']['statuses'] as $status): ?>
                                                <label class="mamba-checkbox">
                                                    <input type="checkbox" 
                                                           name="task[<?php echo esc_attr($taskId); ?>][args][statuses][]" 
                                                           value="<?php echo esc_attr($status); ?>"
                                                           <?php checked(in_array($status, (array)($args['statuses'] ?? []), true)); ?> />
                                                    <?php echo esc_html(ucfirst($status)); ?>
                                                </label>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <?php endforeach; ?>
            </div>

            <div class="mamba-form-actions">
                            <button type="submit" class="mamba-save-button">
                    <span class="dashicons dashicons-saved"></span>
                    <?php echo esc_html( __('Save Settings', 'mamba-cache-for-woocommerce') ); ?>
                </button>
                <button type="button" class="button" id="refresh-counts"
                        data-nonce="<?php echo esc_attr( wp_create_nonce('mamba_db_optimize') ); ?>">
                    <span class="dashicons dashicons-update"></span>
                    <?php echo esc_html( __('Refresh Counts', 'mamba-cache-for-woocommerce') ); ?>
                </button>
            </div>
        </form>
    </div>
</div>

<?php
ob_start();
?>
/* Database Overview Section */
.mamba-db-overview {
    margin-bottom: 20px;
}

.mamba-db-card {
    display: flex;
    align-items: center;
    gap: 20px;
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    width: 100%;
    max-width: -webkit-fill-available;
}

.mamba-db-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.12);
}

.mamba-db-icon {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 48px;
    height: 48px;
    border-radius: 8px;
    background: linear-gradient(135deg, #E68057 0%, #E6704D 100%);
    box-shadow: 0 2px 8px rgba(230, 128, 87, 0.2);
}

.mamba-db-icon .dashicons {
    font-size: 20px;
    width: 20px;
    height: 20px;
    color: #ffffff;
}

.mamba-db-content {
    flex: 1;
}

.mamba-db-title {
    margin: 0 0 6px 0;
    font-size: 16px;
    font-weight: 500;
    color: var(--mamba-dark);
}

.mamba-db-description {
    margin: 0;
    color: var(--mamba-muted);
    line-height: 1.4;
    font-size: 13px;
}

/* Stats Grid for DB */
.mamba-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 16px;
    margin-bottom: 20px;
}

.mamba-stat-card {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 16px;
    transition: all 0.3s ease;
}

.mamba-stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.mamba-stat-header {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 8px;
}

.mamba-stat-header .dashicons {
    color: #E68057;
    font-size: 16px;
    width: 16px;
    height: 16px;
}

.mamba-stat-header h4 {
    margin: 0;
    font-size: 12px;
    font-weight: 500;
    color: var(--mamba-muted);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.mamba-stat-value {
    font-size: 20px;
    font-weight: 600;
    color: var(--mamba-dark);
    line-height: 1.2;
    margin-bottom: 4px;
}

.mamba-stat-description {
    font-size: 11px;
    color: var(--mamba-muted);
}

/* Tasks Grid - Two Column Layout */
.mamba-tasks-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 16px;
    margin-bottom: 20px;
}

@media (max-width: 1200px) {
    .mamba-tasks-grid {
        grid-template-columns: 1fr;
    }
}

/* Task Card */
.mamba-task-card {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 16px;
    transition: all 0.3s ease;
}

.mamba-task-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.mamba-task-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 16px;
    margin-bottom: 12px;
}

.mamba-task-info {
    flex: 1;
}

.mamba-task-name {
    margin: 0 0 4px 0;
    font-size: 14px;
    font-weight: 500;
    color: var(--mamba-dark);
}

.mamba-task-description {
    margin: 0;
    font-size: 12px;
    color: var(--mamba-muted);
    line-height: 1.4;
}

.mamba-task-status {
    display: flex;
    flex-direction: column;
    align-items: flex-end;
    gap: 8px;
}

.mamba-status-badge {
    display: inline-block;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.mamba-status-success {
    background: #dcfce7;
    color: #166534;
}

.mamba-status-warning {
    background: #fef3c7;
    color: #92400e;
}

.mamba-premium-badge-small {
    font-size: 9px;
    padding: 3px 6px;
}

/* Task Details */
.mamba-task-details {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 12px;
    padding-top: 12px;
    border-top: 1px solid #f1f5f9;
}

.mamba-db-detail-item {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.mamba-detail-label {
    font-size: 10px;
    font-weight: 500;
    color: var(--mamba-muted);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.mamba-detail-value {
    font-size: 12px;
    font-weight: 500;
    color: var(--mamba-dark);
}

.mamba-current {
    color: #E68057;
    font-weight: 600;
}

/* Task Actions */
.mamba-task-actions {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-top: 12px;
    padding-top: 12px;
    border-top: 1px solid #f1f5f9;
}

.mamba-run-task {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 6px 12px;
    background: linear-gradient(135deg, #E68057 0%, #E6704D 100%);
    border: none;
    border-radius: 4px;
    font-size: 11px;
    font-weight: 500;
    color: #fff;
    cursor: pointer;
    transition: all 0.2s ease;
}

.mamba-run-task:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(230, 128, 87, 0.3);
}

.mamba-run-task:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.mamba-run-task .dashicons {
    font-size: 12px;
    width: 12px;
    height: 12px;
}

.mamba-task-status-text {
    font-size: 11px;
    color: var(--mamba-muted);
}

.mamba-task-status-text.success {
    color: #16a34a;
}

.mamba-task-status-text.error {
    color: #dc2626;
}

.mamba-task-status-text.loading {
    color: #E68057;
}

/* Settings Row */
.mamba-settings-row {
    margin-top: 12px;
    padding: 12px;
    background: #f8fafc;
    border-radius: 6px;
    display: none;
}

.mamba-settings-row.show {
    display: block;
}

.mamba-settings-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 12px;
}

.mamba-setting-item label {
    display: block;
    font-size: 11px;
    font-weight: 500;
    color: var(--mamba-muted);
    margin-bottom: 4px;
}

.mamba-setting-item input,
.mamba-setting-item select {
    width: 100%;
    padding: 6px 10px;
    border: 1px solid #d1d5db;
    border-radius: 4px;
    font-size: 12px;
}

/* Toggle Switch */
.mamba-toggle-switch {
    display: flex;
    align-items: center;
    justify-content: flex-end;
}

.mamba-toggle-switch input[type="checkbox"] {
    display: none;
}

.mamba-toggle-switch label {
    position: relative;
    padding-left: 40px;
    cursor: pointer;
    font-weight: 500;
    color: var(--mamba-text);
    font-size: 12px;
    user-select: none;
}

.mamba-toggle-switch label::before {
    content: '';
    position: absolute;
    left: 0;
    top: 50%;
    transform: translateY(-50%);
    width: 32px;
    height: 16px;
    background: linear-gradient(135deg, #e5e7eb 0%, #d1d5db 100%);
    border-radius: 4px;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: inset 0 1px 2px rgba(0, 0, 0, 0.1);
}

.mamba-toggle-switch label::after {
    content: '';
    position: absolute;
    left: 2px;
    top: 50%;
    transform: translateY(-50%);
    width: 12px;
    height: 12px;
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border-radius: 3px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.mamba-toggle-switch input[type="checkbox"]:checked + label::before {
    background: linear-gradient(135deg, #E68057 0%, #E6704D 100%);
    box-shadow: inset 0 2px 4px rgba(230, 128, 87, 0.2);
}

.mamba-toggle-switch input[type="checkbox"]:checked + label::after {
    transform: translate(16px, -50%);
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.15);
}

.mamba-toggle-switch input[type="checkbox"]:disabled + label {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Select Styling */
.mamba-select {
    padding: 6px 10px;
    border: 1px solid #d1d5db;
    border-radius: 4px;
    font-size: 12px;
    background: #fff;
    min-width: 120px;
}

.mamba-select:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Dark Mode Overrides */
[data-mamba-theme="dark"] .mamba-db-card,
[data-mamba-theme="dark"] .mamba-stat-card,
[data-mamba-theme="dark"] .mamba-task-card {
    background: var(--mamba-card);
    border-color: var(--mamba-border);
}

[data-mamba-theme="dark"] .mamba-task-details,
[data-mamba-theme="dark"] .mamba-task-actions {
    border-top-color: var(--mamba-border);
}

[data-mamba-theme="dark"] .mamba-settings-row {
    background: rgba(0, 0, 0, 0.2);
}

[data-mamba-theme="dark"] .mamba-select,
[data-mamba-theme="dark"] .mamba-setting-item input,
[data-mamba-theme="dark"] .mamba-setting-item select {
    background: var(--mamba-input-bg);
    border-color: var(--mamba-input-border);
    color: var(--mamba-text-main);
}

[data-mamba-theme="dark"] .mamba-status-success {
    background: rgba(22, 163, 74, 0.2);
    color: #4ade80;
}

[data-mamba-theme="dark"] .mamba-status-warning {
    background: rgba(245, 158, 11, 0.2);
    color: #fbbf24;
}
<?php
$__tab_css = ob_get_clean();
wp_register_style('mamba-tab-db-inline', false);
wp_enqueue_style('mamba-tab-db-inline');
wp_add_inline_style('mamba-tab-db-inline', $__tab_css);
wp_print_styles(['mamba-tab-db-inline']);
?>
