<?php
/**
 * Cache Statistics Service
 *
 * Provides cache hit/miss tracking and statistics recording
 * for monitoring cache performance.
 *
 * @package Mamba\Modules\Caching\Services
 * @since   1.0.0
 */

namespace Mamba\Modules\Caching\Services;

/**
 * Class Stats
 *
 * Records cache hits and misses by content type for performance monitoring.
 * Uses StatsBuffer for efficient batched database writes.
 *
 * @since 1.0.0
 */
final class Stats {
    private static function defaults(): array {
        return [
            'cache_hits'=>0,
            'cache_misses'=>0,
            'cache_size'=>0,
            'last_cleared'=>0,
            'by_type'=>[
                'product'=>['hits'=>0,'misses'=>0],
                'category'=>['hits'=>0,'misses'=>0],
                'shop'=>['hits'=>0,'misses'=>0],
                'store_api'=>['hits'=>0,'misses'=>0],
                'other'=>['hits'=>0,'misses'=>0],
                'homepage'=>['hits'=>0,'misses'=>0], // Added missing homepage type
            ],
            'gen'=>['total_ms'=>0,'misses'=>0],
        ];
    }
    public static function recordHit(string $type = 'other'): void {
        // Use buffered stats instead of immediate database writes
        StatsBuffer::hit($type);
        StatsBuffer::dailyHit($type);
    }
    public static function recordMiss(string $type = 'other', int $generationMs = 0): void {
        // Use buffered stats instead of immediate database writes
        StatsBuffer::miss($type, $generationMs);
        StatsBuffer::dailyMiss($type, $generationMs);
    }
    private static function normalizeType(string $type): string {
        $allowed = ['product','category','shop','store_api','other','homepage'];
        return in_array($type, $allowed, true) ? $type : 'other';
    }


}
