<?php
/**
 * Adaptive TTL Service
 *
 * Premium feature that dynamically adjusts cache TTL based on content characteristics
 * including sales status, modification recency, and content volatility.
 *
 * @package Mamba\Modules\Caching\Services
 * @since   1.0.0
 */

namespace Mamba\Modules\Caching\Services;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class AdaptiveTTL
 *
 * Dynamically adjusts cache TTL based on content characteristics:
 * - Products on sale get shorter TTL (higher volatility)
 * - Recently modified content gets shorter TTL
 * - Evergreen content (unchanged for 30+ days) gets extended TTL
 * - Categories with active sales get adjusted TTL
 * - High-traffic pages can be prioritized for freshness
 *
 * @since 1.0.0
 */
final class AdaptiveTTL {
    
    /**
     * TTL multipliers for different content states
     */
    private const MULTIPLIER_FLASH_SALE = 0.25;      // 25% of base TTL during flash sales
    private const MULTIPLIER_ACTIVE_SALE = 0.50;     // 50% of base TTL during regular sales
    private const MULTIPLIER_RECENTLY_MODIFIED = 0.75; // 75% for content modified in last 7 days
    private const MULTIPLIER_EVERGREEN = 2.0;        // 200% for stable content (30+ days unchanged)
    private const MULTIPLIER_STALE_POPULAR = 1.5;    // 150% for popular evergreen pages
    
    /**
     * Minimum and maximum TTL bounds (in seconds)
     */
    private const MIN_TTL = 300;    // 5 minutes minimum
    private const MAX_TTL = 86400;  // 24 hours maximum
    
    /**
     * Time thresholds (in days)
     */
    private const RECENTLY_MODIFIED_THRESHOLD = 7;   // Days since last modification
    private const EVERGREEN_THRESHOLD = 30;          // Days for content to be considered "stable"
    
    /**
     * Initialize the adaptive TTL system
     */
    public static function init(): void {
        // Register our TTL calculation filter
        add_filter('mamba_cache_ttl_for_request', [__CLASS__, 'calculateAdaptiveTTL'], 10, 2);
        
        // Hook into product save to track modification times
        add_action('woocommerce_update_product', [__CLASS__, 'trackProductModification'], 10, 1);
        
        // Schedule sale boundary checks
        add_action('mamba_check_sale_boundaries', [__CLASS__, 'checkSaleBoundaries']);
        if (!wp_next_scheduled('mamba_check_sale_boundaries')) {
            wp_schedule_event(time(), 'hourly', 'mamba_check_sale_boundaries');
        }
    }
    
    /**
     * Calculate adaptive TTL for a given request context
     * 
     * @param int $baseTTL The base TTL from settings
     * @param array $context Request context (url, post_id, post_type, etc.)
     * @return int Adjusted TTL in seconds
     */
    public static function calculateAdaptiveTTL(int $baseTTL, array $context = []): int {
        // Check if adaptive TTL is enabled
        if (!get_option('mamba_enable_adaptive_ttl', 0)) {
            return $baseTTL;
        }
        
        // Premium feature gate
        if (!function_exists('mamba_fs') || !mamba_fs()->can_use_premium_code__premium_only()) {
            return $baseTTL;
        }
        
        $multiplier = 1.0;
        $reason = 'default';
        
        // Get context information
        $postId = $context['post_id'] ?? 0;
        $postType = $context['post_type'] ?? '';
        $url = $context['url'] ?? '';
        
        // 1. PRODUCT PAGES - Check for sales and modification time
        if ($postType === 'product' && $postId > 0) {
            $productResult = self::calculateProductTTL($postId, $baseTTL);
            $multiplier = $productResult['multiplier'];
            $reason = $productResult['reason'];
        }
        
        // 2. PRODUCT CATEGORY PAGES - Check for active sales in category
        elseif ($postType === 'product_cat' && $postId > 0) {
            $categoryResult = self::calculateCategoryTTL($postId, $baseTTL);
            $multiplier = $categoryResult['multiplier'];
            $reason = $categoryResult['reason'];
        }
        
        // 3. SHOP PAGE - Check for store-wide sales
        elseif (self::isShopPage($url)) {
            $shopResult = self::calculateShopPageTTL($baseTTL);
            $multiplier = $shopResult['multiplier'];
            $reason = $shopResult['reason'];
        }
        
        // 4. PAGES WITH WOOCOMMERCE BLOCKS - Check for block content
        elseif ($postId > 0 && self::containsWooCommerceBlocks($postId)) {
            $blockResult = self::calculateBlockPageTTL($postId, $baseTTL);
            $multiplier = $blockResult['multiplier'];
            $reason = $blockResult['reason'];
        }
        
        // 5. GENERIC PAGES - Check modification time
        elseif ($postId > 0) {
            $pageResult = self::calculateGenericPageTTL($postId, $baseTTL);
            $multiplier = $pageResult['multiplier'];
            $reason = $pageResult['reason'];
        }
        
        // Calculate final TTL with bounds
        $finalTTL = (int) round($baseTTL * $multiplier);
        $finalTTL = max(self::MIN_TTL, min(self::MAX_TTL, $finalTTL));
        
        // Log adaptive TTL decision for debugging
        if (defined('WP_DEBUG') && WP_DEBUG && defined('MAMBA_DEBUG_TTL') && MAMBA_DEBUG_TTL) {
            error_log(sprintf(
                'Mamba Adaptive TTL: base=%d, multiplier=%.2f, final=%d, reason=%s, context=%s',
                $baseTTL,
                $multiplier,
                $finalTTL,
                $reason,
                json_encode($context)
            ));
        }
        
        return $finalTTL;
    }
    
    /**
     * Calculate TTL for a product page
     */
    private static function calculateProductTTL(int $productId, int $baseTTL): array {
        $product = wc_get_product($productId);
        if (!$product) {
            return ['multiplier' => 1.0, 'reason' => 'product_not_found'];
        }
        
        // Check if product is on sale
        if ($product->is_on_sale()) {
            // Check if this is a flash sale (ends within 24 hours)
            $saleEnd = $product->get_date_on_sale_to();
            if ($saleEnd) {
                $hoursUntilEnd = (strtotime($saleEnd->date('Y-m-d H:i:s')) - time()) / 3600;
                
                if ($hoursUntilEnd > 0 && $hoursUntilEnd <= 24) {
                    // Flash sale - very short TTL
                    return ['multiplier' => self::MULTIPLIER_FLASH_SALE, 'reason' => 'flash_sale'];
                }
            }
            
            // Regular sale - reduced TTL
            return ['multiplier' => self::MULTIPLIER_ACTIVE_SALE, 'reason' => 'active_sale'];
        }
        
        // Check modification time
        $lastModified = get_post_modified_time('U', true, $productId);
        if ($lastModified) {
            $daysSinceModified = (time() - $lastModified) / 86400;
            
            if ($daysSinceModified < self::RECENTLY_MODIFIED_THRESHOLD) {
                return ['multiplier' => self::MULTIPLIER_RECENTLY_MODIFIED, 'reason' => 'recently_modified'];
            }
            
            if ($daysSinceModified > self::EVERGREEN_THRESHOLD) {
                // Check if product has low stock (still needs fresh data)
                $stockQty = $product->get_stock_quantity();
                if ($stockQty !== null && $stockQty > 0 && $stockQty <= 5) {
                    return ['multiplier' => 1.0, 'reason' => 'low_stock'];
                }
                
                return ['multiplier' => self::MULTIPLIER_EVERGREEN, 'reason' => 'evergreen_product'];
            }
        }
        
        return ['multiplier' => 1.0, 'reason' => 'default'];
    }
    
    /**
     * Calculate TTL for a category page
     */
    private static function calculateCategoryTTL(int $termId, int $baseTTL): array {
        // Check for active sales in this category
        $saleProductCount = self::countSaleProductsInCategory($termId);
        $totalProductCount = self::countProductsInCategory($termId);
        
        if ($totalProductCount > 0) {
            $salePercentage = ($saleProductCount / $totalProductCount) * 100;
            
            // If more than 50% of products are on sale, treat as flash sale category
            if ($salePercentage >= 50) {
                return ['multiplier' => self::MULTIPLIER_FLASH_SALE, 'reason' => 'flash_sale_category'];
            }
            
            // If 10-50% of products are on sale, use regular sale multiplier
            if ($salePercentage >= 10) {
                return ['multiplier' => self::MULTIPLIER_ACTIVE_SALE, 'reason' => 'sale_category'];
            }
        }
        
        // Check if category has been stable (no product changes recently)
        $recentChanges = self::countRecentChangesInCategory($termId, self::EVERGREEN_THRESHOLD);
        if ($recentChanges === 0 && $totalProductCount > 0) {
            return ['multiplier' => self::MULTIPLIER_EVERGREEN, 'reason' => 'evergreen_category'];
        }
        
        return ['multiplier' => 1.0, 'reason' => 'default'];
    }
    
    /**
     * Calculate TTL for the shop page
     */
    private static function calculateShopPageTTL(int $baseTTL): array {
        // Check for store-wide sales
        $saleProducts = wc_get_products([
            'status' => 'publish',
            'limit' => 1,
            'on_sale' => true,
            'return' => 'ids'
        ]);
        
        if (!empty($saleProducts)) {
            // Check how many products are on sale
            $totalSaleCount = (int) wc_get_products([
                'status' => 'publish',
                'on_sale' => true,
                'return' => 'ids',
                'limit' => -1
            ]);
            
            $totalProductCount = wp_count_posts('product')->publish ?? 0;
            
            if ($totalProductCount > 0) {
                $salePercentage = ($totalSaleCount / $totalProductCount) * 100;
                
                if ($salePercentage >= 30) {
                    return ['multiplier' => self::MULTIPLIER_ACTIVE_SALE, 'reason' => 'store_wide_sale'];
                }
            }
        }
        
        // Shop page benefits from longer cache when stable
        $recentPublished = get_posts([
            'post_type' => 'product',
            'post_status' => 'publish',
            'date_query' => [
                ['after' => self::EVERGREEN_THRESHOLD . ' days ago']
            ],
            'fields' => 'ids',
            'posts_per_page' => 1
        ]);
        
        if (empty($recentPublished)) {
            return ['multiplier' => self::MULTIPLIER_STALE_POPULAR, 'reason' => 'stable_shop'];
        }
        
        return ['multiplier' => 1.0, 'reason' => 'default'];
    }
    
    /**
     * Calculate TTL for generic pages/posts
     */
    private static function calculateGenericPageTTL(int $postId, int $baseTTL): array {
        $lastModified = get_post_modified_time('U', true, $postId);
        if (!$lastModified) {
            return ['multiplier' => 1.0, 'reason' => 'no_modification_date'];
        }
        
        $daysSinceModified = (time() - $lastModified) / 86400;
        
        if ($daysSinceModified < self::RECENTLY_MODIFIED_THRESHOLD) {
            return ['multiplier' => self::MULTIPLIER_RECENTLY_MODIFIED, 'reason' => 'recently_modified'];
        }
        
        if ($daysSinceModified > self::EVERGREEN_THRESHOLD) {
            return ['multiplier' => self::MULTIPLIER_EVERGREEN, 'reason' => 'evergreen_page'];
        }
        
        return ['multiplier' => 1.0, 'reason' => 'default'];
    }
    
    /**
     * Check if URL is the shop page
     */
    private static function isShopPage(string $url): bool {
        if (empty($url)) return false;
        
        $shopUrl = wc_get_page_permalink('shop');
        if (!$shopUrl) return false;
        
        $shopPath = parse_url($shopUrl, PHP_URL_PATH);
        $urlPath = parse_url($url, PHP_URL_PATH);
        
        return rtrim($shopPath, '/') === rtrim($urlPath, '/');
    }
    
    /**
     * Count products on sale in a category
     */
    private static function countSaleProductsInCategory(int $termId): int {
        $args = [
            'status' => 'publish',
            'category' => [get_term($termId)->slug ?? ''],
            'on_sale' => true,
            'return' => 'ids',
            'limit' => -1
        ];
        
        return count(wc_get_products($args));
    }
    
    /**
     * Count total products in a category
     */
    private static function countProductsInCategory(int $termId): int {
        $term = get_term($termId);
        return $term ? (int) $term->count : 0;
    }
    
    /**
     * Count products modified in the last N days in a category
     */
    private static function countRecentChangesInCategory(int $termId, int $days): int {
        $cacheKey = 'mamba_ttl_recent_' . $termId . '_' . $days;
        $cached = wp_cache_get($cacheKey, 'mamba_adaptive_ttl');
        if (false !== $cached) {
            return (int) $cached;
        }
        
        global $wpdb;
        
        $term = get_term($termId);
        if (!$term) return 0;
        
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $count = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->term_relationships} tr ON p.ID = tr.object_id
            INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
            WHERE tt.term_id = %d
            AND p.post_type = 'product'
            AND p.post_status = 'publish'
            AND p.post_modified > %s
        ", $termId, $cutoffDate));
        
        wp_cache_set($cacheKey, $count, 'mamba_adaptive_ttl', 300);
        return $count;
    }
    
    /**
     * Track product modification for adaptive TTL calculations
     */
    public static function trackProductModification(int $productId): void {
        // Update last modification timestamp (WordPress already tracks this, but we can add metadata if needed)
        update_post_meta($productId, '_mamba_last_significant_change', time());
        wp_cache_delete('mamba_ttl_stats', 'mamba_adaptive_ttl');
    }
    
    /**
     * Check for sale boundaries and trigger cache invalidation
     * This runs hourly to catch sales that start or end
     */
    public static function checkSaleBoundaries(): void {
        $now = time();
        $oneHourAgo = $now - 3600;
        $oneHourAhead = $now + 3600;
        
        // Find products where sale just started (within last hour)
        $saleStarted = wc_get_products([
            'status' => 'publish',
            'date_on_sale_from' => date('Y-m-d H:i:s', $oneHourAgo) . '...' . date('Y-m-d H:i:s', $now),
            'return' => 'ids',
            'limit' => -1
        ]);
        
        // Find products where sale ends within the next hour
        $saleEnding = wc_get_products([
            'status' => 'publish',
            'date_on_sale_to' => date('Y-m-d H:i:s', $now) . '...' . date('Y-m-d H:i:s', $oneHourAhead),
            'return' => 'ids',
            'limit' => -1
        ]);
        
        $productsToInvalidate = array_unique(array_merge($saleStarted, $saleEnding));
        
        foreach ($productsToInvalidate as $productId) {
            Invalidation::clearRelatedProduct($productId);
        }
        
        // Log for debugging
        if (!empty($productsToInvalidate) && defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Mamba Adaptive TTL: Sale boundary check invalidated ' . count($productsToInvalidate) . ' products');
        }
    }
    
    /**
     * Get TTL statistics for dashboard display
     */
    public static function getStatistics(): array {
        if (!get_option('mamba_enable_adaptive_ttl', 0)) {
            return [
                'enabled' => false,
                'products_on_sale' => 0,
                'flash_sales' => 0,
                'evergreen_products' => 0,
                'recently_modified' => 0
            ];
        }
        
        $cached = wp_cache_get('mamba_ttl_stats', 'mamba_adaptive_ttl');
        if (false !== $cached) {
            return $cached;
        }
        
        // Count products on sale
        $productsOnSale = count(wc_get_products([
            'status' => 'publish',
            'on_sale' => true,
            'return' => 'ids',
            'limit' => -1
        ]));
        
        // Count flash sales (ending within 24 hours)
        $flashSales = 0;
        $saleProducts = wc_get_products([
            'status' => 'publish',
            'on_sale' => true,
            'limit' => 100
        ]);
        
        foreach ($saleProducts as $product) {
            $saleEnd = $product->get_date_on_sale_to();
            if ($saleEnd) {
                $hoursUntilEnd = (strtotime($saleEnd->date('Y-m-d H:i:s')) - time()) / 3600;
                if ($hoursUntilEnd > 0 && $hoursUntilEnd <= 24) {
                    $flashSales++;
                }
            }
        }
        
        // Count evergreen products (unchanged for 30+ days)
        global $wpdb;
        $cutoffDate = date('Y-m-d H:i:s', strtotime('-' . self::EVERGREEN_THRESHOLD . ' days'));
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $evergreenProducts = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) FROM {$wpdb->posts}
            WHERE post_type = 'product'
            AND post_status = 'publish'
            AND post_modified < %s
        ", $cutoffDate));
        
        // Count recently modified products
        $recentCutoff = date('Y-m-d H:i:s', strtotime('-' . self::RECENTLY_MODIFIED_THRESHOLD . ' days'));
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $recentlyModified = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) FROM {$wpdb->posts}
            WHERE post_type = 'product'
            AND post_status = 'publish'
            AND post_modified >= %s
        ", $recentCutoff));
        
        $stats = [
            'enabled' => true,
            'products_on_sale' => $productsOnSale,
            'flash_sales' => $flashSales,
            'evergreen_products' => $evergreenProducts,
            'recently_modified' => $recentlyModified,
            'multipliers' => [
                'flash_sale' => self::MULTIPLIER_FLASH_SALE,
                'active_sale' => self::MULTIPLIER_ACTIVE_SALE,
                'recently_modified' => self::MULTIPLIER_RECENTLY_MODIFIED,
                'evergreen' => self::MULTIPLIER_EVERGREEN
            ]
        ];
        
        wp_cache_set('mamba_ttl_stats', $stats, 'mamba_adaptive_ttl', 3600);
        
        return $stats;
    }

    /**
     * Calculate TTL for pages containing WooCommerce blocks
     */
    private static function calculateBlockPageTTL(int $postId, int $baseTTL): array {
        // Parse blocks to determine what types are present
        $blocks = self::parseWooCommerceBlocks($postId);
        
        if (empty($blocks)) {
            return ['multiplier' => 1.0, 'reason' => 'no_woo_blocks'];
        }
        
        // Analyze block types and attributes
        $analysis = self::analyzeBlockContent($blocks);
        
        // Apply multipliers based on analysis
        return self::calculateMultiplierFromAnalysis($analysis);
    }

    /**
     * Analyze block content for TTL decision making
     */
    private static function analyzeBlockContent(array $blocks): array {
        $analysis = [
            'has_dynamic_blocks' => false,
            'has_interactive_blocks' => false,
            'has_product_blocks' => false,
            'has_filtered_blocks' => false,
            'has_sorted_blocks' => false,
            'has_recent_blocks' => false,
            'block_count' => count($blocks),
            'product_specific_blocks' => 0,
            'category_specific_blocks' => 0
        ];
        
        foreach ($blocks as $block) {
            $blockName = $block['blockName'] ?? '';
            $attrs = $block['attrs'] ?? [];
            
            // Check block types
            if (self::isDynamicBlock($blockName)) {
                $analysis['has_dynamic_blocks'] = true;
            }
            
            if (self::isInteractiveBlock($blockName)) {
                $analysis['has_interactive_blocks'] = true;
            }
            
            if (self::isProductBlock($blockName)) {
                $analysis['has_product_blocks'] = true;
                
                // Check for product-specific blocks
                if (!empty($attrs['productId']) || !empty($attrs['products'])) {
                    $analysis['product_specific_blocks']++;
                }
            }
            
            if (self::isCategoryBlock($blockName)) {
                // Check for category-specific blocks
                if (!empty($attrs['categoryId']) || !empty($attrs['categories'])) {
                    $analysis['category_specific_blocks']++;
                }
            }
            
            // Check for filtering/sorting attributes
            if (self::hasFilteringAttributes($attrs)) {
                $analysis['has_filtered_blocks'] = true;
            }
            
            if (self::hasSortingAttributes($attrs)) {
                $analysis['has_sorted_blocks'] = true;
            }
            
            if (self::hasRecentAttributes($attrs)) {
                $analysis['has_recent_blocks'] = true;
            }
        }
        
        return $analysis;
    }

    /**
     * Calculate multiplier based on block analysis
     */
    private static function calculateMultiplierFromAnalysis(array $analysis): array {
        // Priority order for TTL reduction
        if ($analysis['has_dynamic_blocks']) {
            return ['multiplier' => self::MULTIPLIER_ACTIVE_SALE, 'reason' => 'dynamic_blocks'];
        }
        
        if ($analysis['has_interactive_blocks'] && $analysis['has_filtered_blocks']) {
            return ['multiplier' => 0.6, 'reason' => 'interactive_filtered_blocks'];
        }
        
        if ($analysis['has_recent_blocks']) {
            return ['multiplier' => 0.7, 'reason' => 'recent_content_blocks'];
        }
        
        if ($analysis['has_sorted_blocks'] || $analysis['has_filtered_blocks']) {
            return ['multiplier' => 0.75, 'reason' => 'sorted_filtered_blocks'];
        }
        
        if ($analysis['has_product_blocks']) {
            // Product-specific blocks are more stable
            if ($analysis['product_specific_blocks'] > 0) {
                return ['multiplier' => 0.85, 'reason' => 'product_specific_blocks'];
            }
            return ['multiplier' => 0.8, 'reason' => 'product_blocks'];
        }
        
        // Generic Woo blocks
        return ['multiplier' => 0.9, 'reason' => 'woo_blocks'];
    }

    /**
     * Check if block is highly dynamic (cart, checkout, etc.)
     */
    private static function isDynamicBlock(string $blockName): bool {
        $dynamicBlocks = [
            'woocommerce/cart',
            'woocommerce/checkout',
            'woocommerce/mini-cart',
            'woocommerce/customer-account'
        ];
        
        return in_array($blockName, $dynamicBlocks);
    }

    /**
     * Check if block is interactive (makes API calls, filters, etc.)
     */
    private static function isInteractiveBlock(string $blockName): bool {
        $interactiveBlocks = [
            'woocommerce/filter-wrapper',
            'woocommerce/attribute-filter',
            'woocommerce/price-filter',
            'woocommerce/stock-filter',
            'woocommerce/product-search',
            'woocommerce/active-filters',
            'woocommerce/product-categories'
        ];
        
        foreach ($interactiveBlocks as $pattern) {
            if (strpos($blockName, $pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if block displays products
     */
    private static function isProductBlock(string $blockName): bool {
        $productBlocks = [
            'woocommerce/products',
            'woocommerce/product',
            'woocommerce/featured-product',
            'woocommerce/product-on-sale',
            'woocommerce/product-new',
            'woocommerce/product-top-rated',
            'woocommerce/product-best-sellers'
        ];
        
        return in_array($blockName, $productBlocks);
    }

    /**
     * Check if block displays categories
     */
    private static function isCategoryBlock(string $blockName): bool {
        $categoryBlocks = [
            'woocommerce/product-category',
            'woocommerce/product-categories',
            'woocommerce/products' // Can have category filters
        ];
        
        return in_array($blockName, $categoryBlocks);
    }

    /**
     * Check if block has filtering attributes
     */
    private static function hasFilteringAttributes(array $attrs): bool {
        $filterKeys = ['categoryId', 'categories', 'tagId', 'tags', 'attributes', 'stockStatus', 'minPrice', 'maxPrice'];
        
        foreach ($filterKeys as $key) {
            if (isset($attrs[$key]) && !empty($attrs[$key])) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if block has sorting attributes
     */
    private static function hasSortingAttributes(array $attrs): bool {
        $sortKeys = ['orderby', 'order'];
        
        foreach ($sortKeys as $key) {
            if (isset($attrs[$key]) && !empty($attrs[$key])) {
                // Default sorting doesn't count as special
                if ($key === 'orderby' && in_array($attrs[$key], ['menu_order', 'title'])) {
                    continue;
                }
                if ($key === 'order' && $attrs[$key] === 'ASC') {
                    continue;
                }
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if block shows recent content
     */
    private static function hasRecentAttributes(array $attrs): bool {
        if (isset($attrs['orderby']) && $attrs['orderby'] === 'date') {
            return true;
        }
        
        // Product "new" blocks show recent additions
        return isset($attrs['columns']) && strpos(($attrs['blockName'] ?? ''), 'product-new') !== false;
    }

    /**
     * Parse WooCommerce blocks from post content
     */
    private static function parseWooCommerceBlocks(int $postId): array {
        $post = get_post($postId);
        if (!$post) {
            return [];
        }
        
        $content = $post->post_content;
        $wooBlocks = [];
        
        // Parse Gutenberg blocks
        if (function_exists('parse_blocks')) {
            $blocks = parse_blocks($content);
            foreach ($blocks as $block) {
                $blockName = $block['blockName'] ?? '';
                if (strpos($blockName, 'woocommerce/') === 0) {
                    $wooBlocks[] = $block;
                }
            }
        }
        
        return $wooBlocks;
    }

    /**
     * Check if a post contains WooCommerce blocks
     */
    private static function containsWooCommerceBlocks(int $postId): bool {
        return !empty(self::parseWooCommerceBlocks($postId));
    }
}
