<?php
/**
 * Caching Module
 *
 * Core caching functionality including full-page cache, Store API cache,
 * cache warmup, tag-based invalidation, and Adaptive TTL.
 *
 * @package Mamba\Modules\Caching
 * @since   1.0.0
 */

namespace Mamba\Modules\Caching;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

use Mamba\Support\Paths;
use Mamba\Modules\Caching\Admin\Actions as AdminActions;

/**
 * Class Module
 *
 * Main caching module that registers controllers, admin actions,
 * and renders caching-related admin tabs.
 *
 * @since 1.0.0
 */
final class Module {
    public function __construct(private Paths $paths) {}
    public function register(): void {
        (new Controller($this->paths))->registerHooks();
        (new AdminActions())->register();
        
        // Initialize Adaptive TTL
        \Mamba\Modules\Caching\Services\AdaptiveTTL::init();

        // Initialize Block Cache Manager
        \Mamba\Modules\Caching\Services\BlockCacheManager::init();
        
        add_action('mamba/admin/tab/caching', [$this, 'renderTab'], 10, 1);
        add_action('mamba/admin/tab/dashboard', [$this, 'renderDashboard'], 10, 1);
        add_action('mamba/admin/tab/settings', [$this, 'renderSettings'], 10, 1); // legacy slug
        add_action('mamba/admin/tab/status', [$this, 'renderSettings'], 10, 1);   // new slug
        add_action('mamba/admin/tab/overhead', [$this, 'renderOverhead'], 10, 1);
        add_action('mamba/admin/tab/savings', [$this, 'renderSavings'], 10, 1);
    }
    public function renderTab(Paths $paths): void {
        $view = __DIR__ . '/views/TabCaching.php';
        if (file_exists($view)) require $view; else echo '<div class="mamba-card">Caching settings here…</div>';
    }
    public function renderDashboard(Paths $paths): void {
        $view = $paths->view('dashboard.php');
        if (file_exists($view)) require $view; else echo '<div class="mamba-card">Dashboard…</div>';
    }
    public function renderSettings(Paths $paths): void {
        $view = $paths->view('settings.php');
        if (file_exists($view)) require $view; else echo '<div class="mamba-card">Settings…</div>';
    }
    public function renderOverhead(Paths $paths): void {
        $view = $paths->moduleView('Caching', 'TabOverhead.php');
        if (file_exists($view)) require $view; else echo '<div class="mamba-card">Overhead controls…</div>';
    }
    public function renderSavings(Paths $paths): void {
        $view = $paths->moduleView('Caching', 'TabSavings.php');
        if (file_exists($view)) require $view; else echo '<div class="mamba-card">Savings & Impact…</div>';
    }
}
