<?php
/**
 * CDN Provider Interface
 *
 * Defines the contract for CDN provider implementations including
 * Cloudflare, Bunny CDN, and generic webhook adapters.
 *
 * @package Mamba\Modules\CDN\Services
 * @since   1.0.0
 */

namespace Mamba\Modules\CDN\Services;

/**
 * Interface Provider
 *
 * Contract for CDN provider implementations defining methods for
 * connection checking, URL purging, tag-based purging, and full cache purging.
 *
 * @since 1.0.0
 */
interface Provider {
    /**
     * Check if the provider is connected and authenticated
     */
    public function isConnected(): bool;
    
    /**
     * Get the provider name
     */
    public function getName(): string;
    
    /**
     * Purge a list of URLs. Optionally include header combinations for APO/custom cache keys.
     * 
     * @param array $urls Array of URLs to purge
     * @param array $headerCombos Array of header combinations for APO/custom cache keys
     */
    public function purgeUrls(array $urls, array $headerCombos = []): Result;
    
    /**
     * Purge by logical groups/tags (Cloudflare Cache-Tag / Bunny CDN-Tag).
     * 
     * @param array $tags Array of tags to purge
     */
    public function purgeTags(array $tags): Result;
    
    /**
     * Full zone purge (discouraged; admin only).
     */
    public function purgeAll(): Result;
    
    /**
     * Optional: apply recommended provider settings (user-initiated).
     */
    public function applyRecommendedSettings(): Result;
    
    /**
     * Test connection to the CDN provider
     */
    public function testConnection(): Result;
}
