<?php
/**
 * Plugin Deactivator
 *
 * Handles plugin deactivation for both single-site and multisite installations.
 * Clears scheduled events and resets welcome modal state.
 *
 * @package Mamba\Lifecycle
 * @since   1.0.0
 */

namespace Mamba\Lifecycle;

/**
 * Class Deactivator
 *
 * Manages plugin deactivation logic including clearing scheduled events
 * and cleaning up temporary states for both single-site and network-wide deactivation.
 *
 * @since 1.0.0
 */
final class Deactivator {
    
    /**
     * Main deactivation entry point
     * Handles both single-site and network-wide deactivation
     * 
     * @param bool $network_wide Whether this is a network-wide deactivation
     */
    public static function run(bool $network_wide = false): void {
        // Handle multisite network-wide deactivation
        if (is_multisite() && $network_wide) {
            self::deactivateNetworkWide();
        } else {
            self::deactivateSingleSite();
        }
    }
    
    /**
     * Deactivate plugin for all sites in the network
     */
    private static function deactivateNetworkWide(): void {
        $sites = get_sites([
            'fields' => 'ids',
            'number' => 0,
        ]);
        
        foreach ($sites as $siteId) {
            switch_to_blog($siteId);
            self::deactivateSingleSite();
            restore_current_blog();
        }
    }
    
    /**
     * Deactivate plugin for a single site
     * This contains all the original deactivation logic
     */
    public static function deactivateSingleSite(): void {
        // Clear cache-related scheduled events
        $timestamp = wp_next_scheduled('mamba_cache_cleanup');
        if ($timestamp) wp_unschedule_event($timestamp, 'mamba_cache_cleanup');
        $timestamp = wp_next_scheduled('mamba_cache_preload');
        if ($timestamp) wp_unschedule_event($timestamp, 'mamba_cache_preload');
        
        // Clear scheduled warmup (Premium feature)
        wp_clear_scheduled_hook('mamba_scheduled_warmup');
        
        // Clear licensing scheduled events
        $timestamp = wp_next_scheduled('mamba_license_daily_validation');
        if ($timestamp) wp_unschedule_event($timestamp, 'mamba_license_daily_validation');
        // Clear background validation lock
        delete_transient('mamba_license_check_lock');
        
        // Clear DB optimizer scheduled events
        if (class_exists('\Mamba\Modules\DB\Services\DatabaseOptimizer')) {
            $optimizer = new \Mamba\Modules\DB\Services\DatabaseOptimizer();
            foreach ($optimizer->getTaskIds() as $taskId) {
                $hook = 'mamba_db_task_' . $taskId;
                wp_clear_scheduled_hook($hook, [$taskId]);
            }
        }
        
        // Clear log cleanup scheduled event
        $timestamp = wp_next_scheduled('mamba_cleanup_logs');
        if ($timestamp) wp_unschedule_event($timestamp, 'mamba_cleanup_logs');
        
        // Clear Critical CSS transient cache
        if (class_exists('\Mamba\Modules\CriticalCss\Services\CriticalCssManager')) {
            $manager = new \Mamba\Modules\CriticalCss\Services\CriticalCssManager();
            $manager->clearCache();
        }
        
        // Reset welcome modal state so it shows again on reactivation
        delete_option('mamba_welcome_shown');
        delete_option('mamba_first_visit_done');
        delete_transient('mamba_activation_redirect');
    }
}
